// State
let isScraping = false;
let shouldStop = false;
let stats = {
    total: 0,
    scraped: 0,
    failed: 0
};

// DOM Elements
const albumSelect = document.getElementById('albumSelect');
const maxPages = document.getElementById('maxPages');
const delay = document.getElementById('delay');
const categoryFilter = document.getElementById('categoryFilter');
const startScrapeBtn = document.getElementById('startScrapeBtn');
const stopScrapeBtn = document.getElementById('stopScrapeBtn');
const viewScrapedBtn = document.getElementById('viewScrapedBtn');
const progressSection = document.getElementById('progressSection');
const progressFill = document.getElementById('progressFill');
const currentStatus = document.getElementById('currentStatus');
const logBox = document.getElementById('logBox');
const postsPreview = document.getElementById('postsPreview');
const postsGrid = document.getElementById('postsGrid');

const totalPostsEl = document.getElementById('totalPosts');
const scrapedPostsEl = document.getElementById('scrapedPosts');
const failedPostsEl = document.getElementById('failedPosts');
const progressPercentEl = document.getElementById('progressPercent');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadAlbums();
    loadStats();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    startScrapeBtn.addEventListener('click', startScraping);
    stopScrapeBtn.addEventListener('click', stopScraping);
    viewScrapedBtn.addEventListener('click', viewScrapedPosts);
}

// Load albums
async function loadAlbums() {
    try {
        const response = await fetch('bulk_scraper_api.php?action=list_albums');
        const albums = await response.json();
        
        albums.forEach(album => {
            const option = document.createElement('option');
            option.value = album.id;
            option.textContent = album.title;
            albumSelect.appendChild(option);
        });
    } catch (error) {
        console.error('Load albums error:', error);
    }
}

// Load stats
async function loadStats() {
    try {
        const response = await fetch('bulk_scraper_api.php?action=get_scraped_posts');
        const posts = await response.json();
        stats.total = posts.length;
        stats.scraped = posts.length;
        updateStats();
    } catch (error) {
        console.error('Load stats error:', error);
    }
}

// Update stats display
function updateStats() {
    totalPostsEl.textContent = stats.total;
    scrapedPostsEl.textContent = stats.scraped;
    failedPostsEl.textContent = stats.failed;
    
    const percent = stats.total > 0 ? Math.round((stats.scraped / stats.total) * 100) : 0;
    progressPercentEl.textContent = percent + '%';
}

// Add log
function addLog(message, type = 'info') {
    const entry = document.createElement('div');
    entry.className = `log-entry ${type}`;
    entry.textContent = `[${new Date().toLocaleTimeString()}] ${message}`;
    logBox.appendChild(entry);
    logBox.scrollTop = logBox.scrollHeight;
}

// Start scraping
async function startScraping() {
    if (isScraping) return;
    
    isScraping = true;
    shouldStop = false;
    
    startScrapeBtn.disabled = true;
    stopScrapeBtn.disabled = false;
    progressSection.classList.add('active');
    logBox.innerHTML = '';
    
    stats.total = 0;
    stats.scraped = 0;
    stats.failed = 0;
    updateStats();
    
    const maxPagesValue = parseInt(maxPages.value) || 10;
    const delayValue = parseInt(delay.value) || 2;
    
    addLog('Bắt đầu scraping...', 'info');
    currentStatus.textContent = 'Đang lấy danh sách posts...';
    
    let allPosts = [];
    let currentPage = 1;
    
    // Collect all posts from pages
    try {
        while (currentPage <= maxPagesValue && !shouldStop) {
            addLog(`Đang lấy trang ${currentPage}/${maxPagesValue}...`, 'info');
            currentStatus.textContent = `Đang lấy trang ${currentPage}/${maxPagesValue}...`;
            
            const response = await fetch(`bulk_scraper_api.php?action=get_posts_list&page=${currentPage}&max_pages=${maxPagesValue}`);
            const result = await response.json();
            
            if (result.error) {
                addLog(`Lỗi: ${result.error}`, 'error');
                break;
            }
            
            if (result.posts && result.posts.length > 0) {
                allPosts = allPosts.concat(result.posts);
                addLog(`Tìm thấy ${result.posts.length} posts trong trang ${currentPage}`, 'success');
                stats.total = allPosts.length;
                updateStats();
            }
            
            if (!result.has_more) {
                addLog('Đã lấy hết tất cả posts', 'success');
                break;
            }
            
            currentPage++;
            
            // Delay between pages
            await sleep(delayValue * 1000);
        }
        
        addLog(`Tổng cộng: ${allPosts.length} posts`, 'success');
        stats.total = allPosts.length;
        updateStats();
        
        // Now scrape each post
        addLog('Bắt đầu scrape chi tiết từng post...', 'info');
        currentStatus.textContent = `Đang scrape ${allPosts.length} posts...`;
        
        for (let i = 0; i < allPosts.length && !shouldStop; i++) {
            const post = allPosts[i];
            const progress = Math.round(((i + 1) / allPosts.length) * 100);
            
            progressFill.style.width = progress + '%';
            progressFill.textContent = progress + '%';
            currentStatus.textContent = `Đang scrape: ${post.title || post.url} (${i + 1}/${allPosts.length})`;
            
            try {
                addLog(`[${i + 1}/${allPosts.length}] Scraping: ${post.title || post.url}`, 'info');
                
                const scrapeResponse = await fetch('bulk_scraper_api.php?action=scrape_post', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        url: post.url,
                        slug: post.slug
                    })
                });
                
                const scrapeResult = await scrapeResponse.json();
                
                if (scrapeResult.success) {
                    stats.scraped++;
                    addLog(`✓ Thành công: ${post.title || post.url}`, 'success');
                } else {
                    stats.failed++;
                    addLog(`✗ Lỗi: ${scrapeResult.error || 'Unknown error'}`, 'error');
                }
                
            } catch (error) {
                stats.failed++;
                addLog(`✗ Lỗi khi scrape: ${error.message}`, 'error');
            }
            
            updateStats();
            
            // Delay between posts
            if (i < allPosts.length - 1) {
                await sleep(delayValue * 1000);
            }
        }
        
        if (shouldStop) {
            addLog('Đã dừng scraping', 'info');
        } else {
            addLog('Hoàn thành!', 'success');
            currentStatus.textContent = `Hoàn thành: ${stats.scraped} thành công, ${stats.failed} lỗi`;
            progressFill.style.width = '100%';
            progressFill.textContent = '100%';
        }
        
    } catch (error) {
        addLog(`Lỗi: ${error.message}`, 'error');
        currentStatus.textContent = 'Có lỗi xảy ra!';
    } finally {
        isScraping = false;
        startScrapeBtn.disabled = false;
        stopScrapeBtn.disabled = true;
    }
}

// Stop scraping
function stopScraping() {
    shouldStop = true;
    addLog('Đang dừng...', 'info');
    stopScrapeBtn.disabled = true;
}

// View scraped posts
async function viewScrapedPosts() {
    try {
        postsPreview.style.display = 'block';
        postsGrid.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Đang tải...</div>';
        
        const response = await fetch('bulk_scraper_api.php?action=get_scraped_posts');
        const posts = await response.json();
        
        if (posts.length === 0) {
            postsGrid.innerHTML = '<div class="empty-state"><p>Chưa có post nào được scrape</p></div>';
            return;
        }
        
        postsGrid.innerHTML = posts.map(post => {
            const thumbnail = post.thumbnail || 'https://via.placeholder.com/300x200?text=No+Image';
            const title = post.title || `${post.cosplayer || ''} cosplay ${post.character || ''}`.trim() || 'Untitled';
            
            return `
                <div class="post-preview-card">
                    <img src="${thumbnail}" alt="${title}" onerror="this.src='https://via.placeholder.com/300x200?text=Error'">
                    <div class="post-preview-info">
                        <h4>${escapeHtml(title.substring(0, 50))}${title.length > 50 ? '...' : ''}</h4>
                        <div class="post-preview-meta">
                            <p><i class="fas fa-user"></i> ${escapeHtml(post.cosplayer || 'N/A')}</p>
                            <p><i class="fas fa-theater-masks"></i> ${escapeHtml(post.character || 'N/A')}</p>
                            <p><i class="fas fa-image"></i> ${post.photo_count || 0} photos</p>
                            <p><i class="fas fa-video"></i> ${post.video_count || 0} videos</p>
                            <div style="margin-top: 10px; display: flex; gap: 5px;">
                                <button class="btn btn-small" onclick="importPost('${post.slug}')">
                                    <i class="fas fa-upload"></i> Import
                                </button>
                                <button class="btn btn-small" style="background: #dc3545;" onclick="deletePost('${post.slug}')">
                                    <i class="fas fa-trash"></i> Xóa
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        // Scroll to preview
        postsPreview.scrollIntoView({ behavior: 'smooth' });
        
    } catch (error) {
        console.error('View scraped posts error:', error);
        postsGrid.innerHTML = '<div class="empty-state"><p>Lỗi khi tải posts</p></div>';
    }
}

// Import post to album
async function importPost(slug) {
    const albumId = albumSelect.value;
    
    if (!albumId) {
        alert('Vui lòng chọn album trước!');
        return;
    }
    
    if (!confirm('Bạn có chắc muốn import post này vào album?')) {
        return;
    }
    
    try {
        const response = await fetch('bulk_scraper_api.php?action=import_to_album', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                slug: slug,
                album_id: albumId
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Import thành công!');
            window.location.href = `album_detail.php?id=${albumId}`;
        } else {
            alert('Lỗi: ' + (result.error || 'Không thể import'));
        }
    } catch (error) {
        console.error('Import error:', error);
        alert('Lỗi khi import: ' + error.message);
    }
}

// Delete scraped post
async function deletePost(slug) {
    if (!confirm('Bạn có chắc muốn xóa post đã scrape này?')) {
        return;
    }
    
    try {
        const response = await fetch(`bulk_scraper_api.php?action=delete_scraped&slug=${slug}`, {
            method: 'POST'
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Xóa thành công!');
            viewScrapedPosts(); // Refresh
        } else {
            alert('Lỗi: ' + (result.error || 'Không thể xóa'));
        }
    } catch (error) {
        console.error('Delete error:', error);
        alert('Lỗi khi xóa: ' + error.message);
    }
}

// Utility functions
function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Make functions global
window.importPost = importPost;
window.deletePost = deletePost;
