// State
let album = null;
let posts = [];
const albumId = new URLSearchParams(window.location.search).get('id');

// DOM Elements
const albumHeader = document.getElementById('albumHeader');
const loadingHeader = document.getElementById('loadingHeader');
const postsGrid = document.getElementById('postsGrid');
const loading = document.getElementById('loading');
const createPostBtn = document.getElementById('createPostBtn');
const postModal = document.getElementById('postModal');
const modalPostContent = document.getElementById('modalPostContent');
const modalClose = document.getElementById('modalClose');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    if (!albumId) {
        window.location.href = 'albums.php';
        return;
    }
    
    loadAlbum();
    loadPosts();
    setupEventListeners();
    
    // Update create post button
    createPostBtn.href = `create_post.php?album_id=${albumId}`;
});

// Event Listeners
function setupEventListeners() {
    modalClose.addEventListener('click', closeModal);
    postModal.addEventListener('click', (e) => {
        if (e.target === postModal) closeModal();
    });
    
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && postModal.classList.contains('show')) {
            closeModal();
        }
    });
}

// Load album info
async function loadAlbum() {
    try {
        const response = await fetch(`albums_api.php?action=get_album&id=${albumId}`);
        if (!response.ok) {
            throw new Error('Failed to load album');
        }
        album = await response.json();
        if (album.error) {
            throw new Error(album.error);
        }
        renderAlbumHeader();
    } catch (error) {
        console.error('Load album error:', error);
        albumHeader.innerHTML = '<div class="empty-state"><i class="fas fa-exclamation-triangle"></i><p>Lỗi khi tải album: ' + error.message + '</p></div>';
    } finally {
        loadingHeader.style.display = 'none';
    }
}

// Render album header
function renderAlbumHeader() {
    if (!album) return;
    
    const thumbnail = album.thumbnail || 'https://via.placeholder.com/800x400?text=No+Image';
    const category = album.category ? `<span class="album-category">${album.category}</span>` : '';
    
    albumHeader.innerHTML = `
        <div class="album-header-content">
            <div class="album-header-image">
                <img src="${thumbnail}" alt="${album.title}" onerror="this.src='https://via.placeholder.com/800x400?text=No+Image'">
            </div>
            <div class="album-header-info">
                <h1>${escapeHtml(album.title)} ${category}</h1>
                ${album.description ? `<p class="album-description">${escapeHtml(album.description)}</p>` : ''}
                <div class="album-stats">
                    <span><i class="fas fa-images"></i> <span id="postCount">0</span> posts</span>
                    <span><i class="fas fa-calendar"></i> ${formatDate(album.createdDate)}</span>
                </div>
            </div>
        </div>
    `;
}

// Load posts
async function loadPosts() {
    loading.style.display = 'block';
    postsGrid.innerHTML = '';
    postsGrid.appendChild(loading);
    
    try {
        const response = await fetch(`albums_api.php?action=list_posts&album_id=${albumId}`);
        if (!response.ok) {
            throw new Error('Failed to load posts');
        }
        posts = await response.json();
        if (Array.isArray(posts)) {
            renderPosts();
            // Update post count in header
            const postCountEl = document.getElementById('postCount');
            if (postCountEl) {
                postCountEl.textContent = posts.length;
            }
        } else {
            throw new Error('Invalid response format');
        }
    } catch (error) {
        console.error('Load posts error:', error);
        postsGrid.innerHTML = '<div class="empty-state"><i class="fas fa-exclamation-triangle"></i><p>Lỗi khi tải posts: ' + error.message + '</p></div>';
    } finally {
        loading.style.display = 'none';
    }
}

// Render posts
function renderPosts() {
    if (posts.length === 0) {
        postsGrid.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-image"></i>
                <p>Chưa có post nào trong album này. Hãy tạo post mới!</p>
            </div>
        `;
        return;
    }
    
    postsGrid.innerHTML = posts.map((post, index) => {
        const thumbnail = post.thumbnail || (post.images && post.images.length > 0 ? post.images[0] : 'https://via.placeholder.com/300x200?text=No+Image');
        const imageCount = post.photo_count || (post.images ? post.images.length : 0);
        const videoCount = post.video_count || (post.videos ? post.videos.length : 0);
        const title = post.title || `${post.cosplayer ? post.cosplayer : ''} ${post.character ? 'as ' + post.character : ''}`.trim() || 'Untitled Post';
        
        return `
        <div class="post-card" onclick="openPostModal(${index})">
            <div class="post-thumbnail">
                <img src="${thumbnail}" alt="${title}" onerror="this.src='https://via.placeholder.com/300x200?text=No+Image'">
                <div class="post-overlay">
                    <div class="post-counts">
                        ${imageCount > 0 ? `<span><i class="fas fa-image"></i> ${imageCount}</span>` : ''}
                        ${videoCount > 0 ? `<span><i class="fas fa-video"></i> ${videoCount}</span>` : ''}
                    </div>
                </div>
            </div>
            <div class="post-info">
                <h3 class="post-title">${escapeHtml(title)}</h3>
                ${post.cosplayer ? `<p class="post-meta"><i class="fas fa-user"></i> ${escapeHtml(post.cosplayer)}</p>` : ''}
                ${post.character ? `<p class="post-meta"><i class="fas fa-theater-masks"></i> ${escapeHtml(post.character)}</p>` : ''}
                ${post.source ? `<p class="post-meta"><i class="fas fa-film"></i> ${escapeHtml(post.source)}</p>` : ''}
                ${post.tags && post.tags.length > 0 ? `<div class="post-tags">${post.tags.map(tag => `<span class="tag">${escapeHtml(tag)}</span>`).join('')}</div>` : ''}
            </div>
        </div>
        `;
    }).join('');
}

// Open post modal
function openPostModal(index) {
    const post = posts[index];
    if (!post) return;
    
    const title = post.title || `${post.cosplayer ? post.cosplayer : ''} ${post.character ? 'as ' + post.character : ''}`.trim() || 'Untitled Post';
    
    // Build download links
    let downloadLinksHtml = '';
    if (post.download_links && post.download_links.length > 0) {
        downloadLinksHtml = `
            <div class="download-links" style="margin-top: 15px;">
                <h4 style="margin-bottom: 10px;"><i class="fas fa-download"></i> Download Links:</h4>
                <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                    ${post.download_links.map(link => `
                        <a href="${link.url}" target="_blank" class="btn btn-small" style="background: #28a745; color: white;">
                            <i class="fas fa-external-link-alt"></i> ${link.type ? link.type.charAt(0).toUpperCase() + link.type.slice(1) : 'Download'}
                        </a>
                    `).join('')}
                </div>
            </div>
        `;
    }
    
    // Build source URL
    let sourceUrlHtml = '';
    if (post.source_url) {
        sourceUrlHtml = `
            <div style="margin-top: 10px;">
                <a href="${post.source_url}" target="_blank" class="btn btn-small" style="background: #17a2b8; color: white;">
                    <i class="fas fa-link"></i> Xem gốc
                </a>
            </div>
        `;
    }
    
    let content = `
        <div class="post-modal-header">
            <h2>${escapeHtml(title)}</h2>
            ${post.cosplayer || post.character || post.source ? `
                <div class="post-modal-meta">
                    ${post.cosplayer ? `<span><i class="fas fa-user"></i> ${escapeHtml(post.cosplayer)}</span>` : ''}
                    ${post.character ? `<span><i class="fas fa-theater-masks"></i> ${escapeHtml(post.character)}</span>` : ''}
                    ${post.source ? `<span><i class="fas fa-film"></i> ${escapeHtml(post.source)}</span>` : ''}
                    ${post.photo_count ? `<span><i class="fas fa-image"></i> ${post.photo_count} photos</span>` : ''}
                    ${post.video_count ? `<span><i class="fas fa-video"></i> ${post.video_count} videos</span>` : ''}
                </div>
            ` : ''}
            ${post.description ? `<p class="post-modal-description">${escapeHtml(post.description)}</p>` : ''}
            ${post.tags && post.tags.length > 0 ? `<div class="post-modal-tags">${post.tags.map(tag => `<span class="tag">${escapeHtml(tag)}</span>`).join('')}</div>` : ''}
            ${downloadLinksHtml}
            ${sourceUrlHtml}
        </div>
        <div class="post-modal-content">
    `;
    
    // Add images
    if (post.images && post.images.length > 0) {
        content += '<div class="post-images">';
        post.images.forEach(image => {
            content += `<div class="post-image-item"><img src="${image}" alt="Post image" onerror="this.src=\'https://via.placeholder.com/800x600?text=Image+Error\'"></div>`;
        });
        content += '</div>';
    }
    
    // Add videos
    if (post.videos && post.videos.length > 0) {
        content += '<div class="post-videos">';
        post.videos.forEach(video => {
            const isYoutube = video.includes('youtube.com') || video.includes('youtu.be');
            if (isYoutube) {
                const videoId = extractYoutubeId(video);
                content += `<div class="post-video-item"><iframe src="https://www.youtube.com/embed/${videoId}" frameborder="0" allowfullscreen></iframe></div>`;
            } else {
                content += `<div class="post-video-item"><video controls><source src="${video}" type="video/mp4">Trình duyệt không hỗ trợ video.</video></div>`;
            }
        });
        content += '</div>';
    }
    
    content += '</div>';
    
    modalPostContent.innerHTML = content;
    postModal.classList.add('show');
    document.body.style.overflow = 'hidden';
}

// Close modal
function closeModal() {
    postModal.classList.remove('show');
    document.body.style.overflow = '';
}

// Extract YouTube video ID
function extractYoutubeId(url) {
    const regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|&v=)([^#&?]*).*/;
    const match = url.match(regExp);
    return (match && match[2].length === 11) ? match[2] : null;
}

// Utility functions
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('vi-VN', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

// Make functions global
window.openPostModal = openPostModal;
