<?php
header('Content-Type: application/json');

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'list_albums':
        listAlbums();
        break;
    case 'get_album':
        getAlbum();
        break;
    case 'create_album':
        createAlbum();
        break;
    case 'create_album_with_post':
        createAlbumWithPost();
        break;
    case 'update_album':
        updateAlbum();
        break;
    case 'delete_album':
        deleteAlbum();
        break;
    case 'list_posts':
        listPosts();
        break;
    case 'get_post':
        getPost();
        break;
    case 'create_post':
        createPost();
        break;
    case 'update_post':
        updatePost();
        break;
    case 'delete_post':
        deletePost();
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
        break;
}

/**
 * Lấy danh sách tất cả albums
 */
function listAlbums() {
    $albumsDir = __DIR__ . '/albums/';
    
    if (!is_dir($albumsDir)) {
        mkdir($albumsDir, 0755, true);
        echo json_encode([]);
        return;
    }
    
    $albums = [];
    $files = scandir($albumsDir);
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || pathinfo($file, PATHINFO_EXTENSION) !== 'json') {
            continue;
        }
        
        $filepath = $albumsDir . $file;
        $data = json_decode(file_get_contents($filepath), true);
        
        if ($data && isset($data['id'])) {
            // Đếm số posts
            $postsDir = $albumsDir . $data['id'] . '/posts/';
            $postCount = 0;
            if (is_dir($postsDir)) {
                $postFiles = scandir($postsDir);
                $postCount = count(array_filter($postFiles, function($f) {
                    return pathinfo($f, PATHINFO_EXTENSION) === 'json';
                }));
            }
            
            $albums[] = [
                'id' => $data['id'],
                'title' => $data['title'] ?? 'Untitled Album',
                'description' => $data['description'] ?? '',
                'thumbnail' => $data['thumbnail'] ?? '',
                'category' => $data['category'] ?? '',
                'createdDate' => $data['createdDate'] ?? date('c'),
                'updatedDate' => $data['updatedDate'] ?? date('c'),
                'postCount' => $postCount
            ];
        }
    }
    
    // Sắp xếp theo ngày tạo (mới nhất trước)
    usort($albums, function($a, $b) {
        return strtotime($b['createdDate']) - strtotime($a['createdDate']);
    });
    
    echo json_encode($albums);
}

/**
 * Lấy thông tin một album cụ thể
 */
function getAlbum() {
    $albumId = $_GET['id'] ?? '';
    
    if (empty($albumId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $albumFile = $albumsDir . $albumId . '.json';
    
    if (!file_exists($albumFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy album']);
        return;
    }
    
    $data = json_decode(file_get_contents($albumFile), true);
    echo json_encode($data);
}

/**
 * Tạo album mới
 */
function createAlbum() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    // Support both JSON and form data
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (strpos($contentType, 'application/json') !== false) {
        $data = json_decode(file_get_contents('php://input'), true);
    } else {
        $data = $_POST;
    }
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid data']);
        return;
    }
    
    $title = $data['title'] ?? '';
    $description = $data['description'] ?? '';
    $thumbnail = $data['thumbnail'] ?? '';
    $category = $data['category'] ?? '';
    
    if (empty($title)) {
        http_response_code(400);
        echo json_encode(['error' => 'Tiêu đề album không được để trống']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    if (!is_dir($albumsDir)) {
        mkdir($albumsDir, 0755, true);
    }
    
    $albumId = time() . '-' . uniqid();
    $albumFile = $albumsDir . $albumId . '.json';
    
    // Tạo thư mục cho posts
    $postsDir = $albumsDir . $albumId . '/posts/';
    if (!is_dir($postsDir)) {
        mkdir($postsDir, 0755, true);
    }
    
    $albumData = [
        'id' => $albumId,
        'title' => $title,
        'description' => $description,
        'thumbnail' => $thumbnail,
        'category' => $category,
        'createdDate' => date('c'),
        'updatedDate' => date('c')
    ];
    
    file_put_contents($albumFile, json_encode($albumData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'album' => $albumData
    ]);
}

/**
 * Tạo album mới kèm post (tất cả ảnh)
 */
function createAlbumWithPost() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    // Support both JSON and form data
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (strpos($contentType, 'application/json') !== false) {
        $data = json_decode(file_get_contents('php://input'), true);
    } else {
        $data = $_POST;
    }
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid data']);
        return;
    }
    
    $title = $data['title'] ?? '';
    $thumbnail = $data['thumbnail'] ?? '';
    $category = $data['category'] ?? '';
    $images = $data['images'] ?? [];
    $videos = $data['videos'] ?? [];
    $cosplayer = $data['cosplayer'] ?? '';
    $character = $data['character'] ?? '';
    $source = $data['source'] ?? '';
    $tags = $data['tags'] ?? [];
    $photoCount = $data['photo_count'] ?? count($images);
    $videoCount = $data['video_count'] ?? count($videos);
    
    if (empty($title)) {
        http_response_code(400);
        echo json_encode(['error' => 'Tên album không được để trống']);
        return;
    }
    
    if (empty($thumbnail)) {
        http_response_code(400);
        echo json_encode(['error' => 'Ảnh nền không được để trống']);
        return;
    }
    
    // Validate có ít nhất 1 ảnh hoặc 1 video
    if (empty($images) && empty($videos)) {
        http_response_code(400);
        echo json_encode(['error' => 'Phải có ít nhất 1 ảnh hoặc 1 video']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    if (!is_dir($albumsDir)) {
        mkdir($albumsDir, 0755, true);
    }
    
    $albumId = time() . '-' . uniqid();
    $albumFile = $albumsDir . $albumId . '.json';
    
    // Tạo thư mục cho posts
    $postsDir = $albumsDir . $albumId . '/posts/';
    if (!is_dir($postsDir)) {
        mkdir($postsDir, 0755, true);
    }
    
    // Create album
    $albumData = [
        'id' => $albumId,
        'title' => $title,
        'description' => '', // Không cần description
        'thumbnail' => $thumbnail,
        'category' => $category,
        'createdDate' => date('c'),
        'updatedDate' => date('c')
    ];
    
    file_put_contents($albumFile, json_encode($albumData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    // Create post with all images/videos
    $postId = time() . '-' . uniqid();
    $postFile = $postsDir . $postId . '.json';
    
    // Generate title for post
    $postTitle = '';
    if ($cosplayer && $character) {
        $postTitle = $cosplayer . ' cosplay ' . $character;
        if ($source) {
            $postTitle .= ' – ' . $source;
        }
    } else {
        $postTitle = $title; // Use album title as fallback
    }
    
    $postData = [
        'id' => $postId,
        'album_id' => $albumId,
        'title' => $postTitle,
        'description' => '',
        'images' => $images,
        'videos' => $videos,
        'cosplayer' => $cosplayer,
        'character' => $character,
        'source' => $source,
        'tags' => is_array($tags) ? $tags : [],
        'photo_count' => (int)$photoCount,
        'video_count' => (int)$videoCount,
        'thumbnail' => !empty($images) ? $images[0] : (!empty($videos) ? $videos[0] : $thumbnail),
        'createdDate' => date('c'),
        'updatedDate' => date('c')
    ];
    
    file_put_contents($postFile, json_encode($postData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'album' => $albumData,
        'post' => $postData
    ]);
}

/**
 * Cập nhật album
 */
function updateAlbum() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST' && $_SERVER['REQUEST_METHOD'] !== 'PUT') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    $albumId = $_GET['id'] ?? $_POST['id'] ?? '';
    
    if (empty($albumId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    if (!$data) {
        $data = $_POST;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $albumFile = $albumsDir . $albumId . '.json';
    
    if (!file_exists($albumFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy album']);
        return;
    }
    
    $albumData = json_decode(file_get_contents($albumFile), true);
    
    if (isset($data['title'])) $albumData['title'] = $data['title'];
    if (isset($data['description'])) $albumData['description'] = $data['description'];
    if (isset($data['thumbnail'])) $albumData['thumbnail'] = $data['thumbnail'];
    if (isset($data['category'])) $albumData['category'] = $data['category'];
    
    $albumData['updatedDate'] = date('c');
    
    file_put_contents($albumFile, json_encode($albumData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'album' => $albumData
    ]);
}

/**
 * Xóa album
 */
function deleteAlbum() {
    if ($_SERVER['REQUEST_METHOD'] !== 'DELETE' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    $albumId = $_GET['id'] ?? $_POST['id'] ?? '';
    
    if (empty($albumId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $albumFile = $albumsDir . $albumId . '.json';
    
    if (!file_exists($albumFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy album']);
        return;
    }
    
    // Xóa thư mục posts và album
    $albumDir = $albumsDir . $albumId;
    if (is_dir($albumDir)) {
        deleteDirectory($albumDir);
    }
    
    if (unlink($albumFile)) {
        echo json_encode(['success' => true, 'message' => 'Xóa album thành công']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Không thể xóa album']);
    }
}

/**
 * Lấy danh sách posts trong album
 */
function listPosts() {
    $albumId = $_GET['album_id'] ?? '';
    
    if (empty($albumId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $postsDir = $albumsDir . $albumId . '/posts/';
    
    if (!is_dir($postsDir)) {
        echo json_encode([]);
        return;
    }
    
    $posts = [];
    $files = scandir($postsDir);
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || pathinfo($file, PATHINFO_EXTENSION) !== 'json') {
            continue;
        }
        
        $filepath = $postsDir . $file;
        $data = json_decode(file_get_contents($filepath), true);
        
        if ($data && isset($data['id'])) {
            $posts[] = $data;
        }
    }
    
    // Sắp xếp theo ngày tạo (mới nhất trước)
    usort($posts, function($a, $b) {
        return strtotime($b['createdDate'] ?? '') - strtotime($a['createdDate'] ?? '');
    });
    
    echo json_encode($posts);
}

/**
 * Lấy thông tin một post cụ thể
 */
function getPost() {
    $albumId = $_GET['album_id'] ?? '';
    $postId = $_GET['post_id'] ?? '';
    
    if (empty($albumId) || empty($postId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album hoặc post']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $postFile = $albumsDir . $albumId . '/posts/' . $postId . '.json';
    
    if (!file_exists($postFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy post']);
        return;
    }
    
    $data = json_decode(file_get_contents($postFile), true);
    echo json_encode($data);
}

/**
 * Tạo post mới
 */
function createPost() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    // Support both JSON and form data
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (strpos($contentType, 'application/json') !== false) {
        $data = json_decode(file_get_contents('php://input'), true);
    } else {
        $data = $_POST;
        // Handle arrays from form data
        if (isset($data['images']) && is_string($data['images'])) {
            $data['images'] = json_decode($data['images'], true) ?? [];
        }
        if (isset($data['videos']) && is_string($data['videos'])) {
            $data['videos'] = json_decode($data['videos'], true) ?? [];
        }
        if (isset($data['tags']) && is_string($data['tags'])) {
            $data['tags'] = json_decode($data['tags'], true) ?? [];
        }
    }
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid data']);
        return;
    }
    
    $albumId = $data['album_id'] ?? '';
    $title = $data['title'] ?? '';
    $description = $data['description'] ?? '';
    $images = $data['images'] ?? [];
    $videos = $data['videos'] ?? [];
    $cosplayer = $data['cosplayer'] ?? '';
    $character = $data['character'] ?? '';
    $source = $data['source'] ?? '';
    $tags = $data['tags'] ?? [];
    $downloadLinks = $data['download_links'] ?? [];
    $photoCount = $data['photo_count'] ?? count($images);
    $videoCount = $data['video_count'] ?? count($videos);
    $sourceUrl = $data['source_url'] ?? '';
    $categories = $data['categories'] ?? [];
    
    if (empty($albumId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album']);
        return;
    }
    
    // Validate album exists
    $albumsDir = __DIR__ . '/albums/';
    $albumFile = $albumsDir . $albumId . '.json';
    
    if (!file_exists($albumFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy album']);
        return;
    }
    
    // Validate có ít nhất 1 ảnh hoặc 1 video
    if (empty($images) && empty($videos)) {
        http_response_code(400);
        echo json_encode(['error' => 'Post phải có ít nhất 1 ảnh hoặc 1 video']);
        return;
    }
    
    $postsDir = $albumsDir . $albumId . '/posts/';
    if (!is_dir($postsDir)) {
        mkdir($postsDir, 0755, true);
    }
    
    $postId = time() . '-' . uniqid();
    $postFile = $postsDir . $postId . '.json';
    
    $postData = [
        'id' => $postId,
        'album_id' => $albumId,
        'title' => $title,
        'description' => $description,
        'images' => $images, // Array of image URLs
        'videos' => $videos, // Array of video URLs
        'cosplayer' => $cosplayer,
        'character' => $character,
        'source' => $source,
        'tags' => is_array($tags) ? $tags : [],
        'download_links' => is_array($downloadLinks) ? $downloadLinks : [],
        'photo_count' => (int)$photoCount,
        'video_count' => (int)$videoCount,
        'source_url' => $sourceUrl,
        'categories' => is_array($categories) ? $categories : [],
        'createdDate' => date('c'),
        'updatedDate' => date('c')
    ];
    
    // Set thumbnail (first image or first video thumbnail)
    if (!empty($images)) {
        $postData['thumbnail'] = $images[0];
    } else if (!empty($videos)) {
        $postData['thumbnail'] = $videos[0]; // You might want to extract thumbnail from video
    }
    
    file_put_contents($postFile, json_encode($postData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    // Update album's updatedDate
    $albumData = json_decode(file_get_contents($albumFile), true);
    $albumData['updatedDate'] = date('c');
    file_put_contents($albumFile, json_encode($albumData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'post' => $postData
    ]);
}

/**
 * Cập nhật post
 */
function updatePost() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST' && $_SERVER['REQUEST_METHOD'] !== 'PUT') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    $albumId = $_GET['album_id'] ?? $_POST['album_id'] ?? '';
    $postId = $_GET['post_id'] ?? $_POST['post_id'] ?? '';
    
    if (empty($albumId) || empty($postId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album hoặc post']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    if (!$data) {
        $data = $_POST;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $postFile = $albumsDir . $albumId . '/posts/' . $postId . '.json';
    
    if (!file_exists($postFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy post']);
        return;
    }
    
    $postData = json_decode(file_get_contents($postFile), true);
    
    if (isset($data['title'])) $postData['title'] = $data['title'];
    if (isset($data['description'])) $postData['description'] = $data['description'];
    if (isset($data['images'])) $postData['images'] = $data['images'];
    if (isset($data['videos'])) $postData['videos'] = $data['videos'];
    if (isset($data['cosplayer'])) $postData['cosplayer'] = $data['cosplayer'];
    if (isset($data['character'])) $postData['character'] = $data['character'];
    if (isset($data['source'])) $postData['source'] = $data['source'];
    if (isset($data['tags'])) $postData['tags'] = is_array($data['tags']) ? $data['tags'] : [];
    if (isset($data['download_links'])) $postData['download_links'] = is_array($data['download_links']) ? $data['download_links'] : [];
    if (isset($data['photo_count'])) $postData['photo_count'] = (int)$data['photo_count'];
    if (isset($data['video_count'])) $postData['video_count'] = (int)$data['video_count'];
    if (isset($data['source_url'])) $postData['source_url'] = $data['source_url'];
    if (isset($data['categories'])) $postData['categories'] = is_array($data['categories']) ? $data['categories'] : [];
    
    // Update thumbnail
    if (!empty($postData['images'])) {
        $postData['thumbnail'] = $postData['images'][0];
    } else if (!empty($postData['videos'])) {
        $postData['thumbnail'] = $postData['videos'][0];
    }
    
    $postData['updatedDate'] = date('c');
    
    file_put_contents($postFile, json_encode($postData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'post' => $postData
    ]);
}

/**
 * Xóa post
 */
function deletePost() {
    if ($_SERVER['REQUEST_METHOD'] !== 'DELETE' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    $albumId = $_GET['album_id'] ?? $_POST['album_id'] ?? '';
    $postId = $_GET['post_id'] ?? $_POST['post_id'] ?? '';
    
    if (empty($albumId) || empty($postId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Thiếu ID album hoặc post']);
        return;
    }
    
    $albumsDir = __DIR__ . '/albums/';
    $postFile = $albumsDir . $albumId . '/posts/' . $postId . '.json';
    
    if (!file_exists($postFile)) {
        http_response_code(404);
        echo json_encode(['error' => 'Không tìm thấy post']);
        return;
    }
    
    if (unlink($postFile)) {
        echo json_encode(['success' => true, 'message' => 'Xóa post thành công']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Không thể xóa post']);
    }
}

/**
 * Helper function để xóa thư mục đệ quy
 */
function deleteDirectory($dir) {
    if (!is_dir($dir)) {
        return false;
    }
    
    $files = array_diff(scandir($dir), array('.', '..'));
    
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        if (is_dir($path)) {
            deleteDirectory($path);
        } else {
            unlink($path);
        }
    }
    
    return rmdir($dir);
}
?>
