// State management
let selectedFiles = [];
let selectedFilesPairs = []; // Mảng các cặp ảnh

// DOM Elements
const uploadBoxMultiple = document.getElementById('uploadBoxMultiple');
const fileInputMultiple = document.getElementById('fileInputMultiple');
const previewMultiple = document.getElementById('previewMultiple');
const previewUrl1 = document.getElementById('previewUrl1');
const previewUrl2 = document.getElementById('previewUrl2');
const urlInput1 = document.getElementById('urlInput1');
const urlInput2 = document.getElementById('urlInput2');
const uploadBtn = document.getElementById('uploadBtn');
const uploadUrlBtn = document.getElementById('uploadUrlBtn');
const uploadProgress = document.getElementById('uploadProgress');
const progressFill = document.getElementById('progressFill');
const progressText = document.getElementById('progressText');
const tabBtns = document.querySelectorAll('.tab-btn');
const tabContents = document.querySelectorAll('.upload-tab-content');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    // Tab switching
    tabBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            const tab = btn.dataset.tab;
            switchTab(tab);
        });
    });
    
    // Multiple file upload
    if (uploadBoxMultiple && fileInputMultiple) {
        uploadBoxMultiple.addEventListener('click', () => fileInputMultiple.click());
        
        uploadBoxMultiple.addEventListener('dragover', (e) => handleDragOver(e, uploadBoxMultiple));
        uploadBoxMultiple.addEventListener('drop', (e) => handleDropMultiple(e));
        uploadBoxMultiple.addEventListener('dragleave', (e) => handleDragLeave(e, uploadBoxMultiple));
        
        fileInputMultiple.addEventListener('change', (e) => handleFileSelectMultiple(e));
    }
    
    // URL inputs
    if (urlInput1 && urlInput2) {
        urlInput1.addEventListener('input', () => previewImageFromUrl(urlInput1.value, previewUrl1));
        urlInput2.addEventListener('input', () => previewImageFromUrl(urlInput2.value, previewUrl2));
    }
    
    // Upload buttons
    if (uploadBtn) uploadBtn.addEventListener('click', handleMultipleFileUpload);
    if (uploadUrlBtn) uploadUrlBtn.addEventListener('click', handleUrlUpload);
}

// Tab switching
function switchTab(tab) {
    tabBtns.forEach(btn => {
        if (btn.dataset.tab === tab) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    tabContents.forEach(content => {
        if (content.id === tab + 'Tab') {
            content.classList.add('active');
        } else {
            content.classList.remove('active');
        }
    });
}

// Drag and Drop Multiple
function handleDragOver(e, box) {
    e.preventDefault();
    e.stopPropagation();
    box.style.background = '#e8ebff';
}

function handleDragLeave(e, box) {
    e.preventDefault();
    e.stopPropagation();
    box.style.background = '#fff5f7';
}

function handleDropMultiple(e) {
    e.preventDefault();
    e.stopPropagation();
    const box = e.currentTarget;
    box.style.background = '#fff5f7';
    
    const files = Array.from(e.dataTransfer.files).filter(file => file.type.startsWith('image/'));
    if (files.length > 0) {
        addFilesToSelection(files);
    }
}

function handleFileSelectMultiple(e) {
    const files = Array.from(e.target.files).filter(file => file.type.startsWith('image/'));
    if (files.length > 0) {
        addFilesToSelection(files);
    }
}

function addFilesToSelection(files) {
    files.forEach(file => {
        selectedFiles.push(file);
    });
    updatePreviewMultiple();
    updateUploadButton();
}

function removeFile(index) {
    selectedFiles.splice(index, 1);
    updatePreviewMultiple();
    updateUploadButton();
}

function updatePreviewMultiple() {
    if (!previewMultiple) return;
    
    if (selectedFiles.length === 0) {
        previewMultiple.innerHTML = '';
        return;
    }
    
    // Tạo HTML trước
    previewMultiple.innerHTML = selectedFiles.map((file, index) => {
        const isOdd = (index + 1) % 2 === 1;
        const label = isOdd ? 'Ảnh gốc' : 'Ảnh khi bấm mắt';
        
        return `
            <div class="preview-multiple-item" data-index="${index}">
                <img src="" alt="Preview" style="display: none;">
                <div class="preview-label">${index + 1}. ${label}</div>
                <button class="remove-btn" onclick="removeFileFromPreview(${index})" title="Xóa">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
    }).join('');
    
    // Load images sau khi HTML đã được render
    selectedFiles.forEach((file, index) => {
        const reader = new FileReader();
        reader.onload = (e) => {
            const item = previewMultiple.querySelector(`[data-index="${index}"]`);
            if (item) {
                const img = item.querySelector('img');
                if (img) {
                    img.src = e.target.result;
                    img.style.display = 'block';
                }
            }
        };
        reader.readAsDataURL(file);
    });
}

// Make functions global for onclick handlers
window.removeFileFromPreview = function(index) {
    removeFile(index);
};

function previewImageFromUrl(url, previewBox) {
    if (!url || !url.trim() || !previewBox) {
        if (previewBox) {
            previewBox.innerHTML = '';
            previewBox.classList.remove('has-image');
        }
        return;
    }
    
    // Validate URL
    try {
        new URL(url);
        previewBox.innerHTML = `<img src="${url}" alt="Preview" onerror="this.parentElement.innerHTML=''; this.parentElement.classList.remove('has-image');">`;
        previewBox.classList.add('has-image');
    } catch (e) {
        previewBox.innerHTML = '';
        previewBox.classList.remove('has-image');
    }
}

function updateUploadButton() {
    if (uploadBtn) {
        // Cần số lượng ảnh chẵn (ít nhất 2 ảnh)
        if (selectedFiles.length >= 2 && selectedFiles.length % 2 === 0) {
            uploadBtn.disabled = false;
            uploadBtn.innerHTML = `<i class="fas fa-upload"></i> Upload ${selectedFiles.length / 2} Cặp Ảnh lên Cloudflare R2`;
        } else {
            uploadBtn.disabled = true;
            if (selectedFiles.length > 0) {
                uploadBtn.innerHTML = `<i class="fas fa-upload"></i> Cần số lượng ảnh chẵn (hiện có: ${selectedFiles.length})`;
            } else {
                uploadBtn.innerHTML = `<i class="fas fa-upload"></i> Upload Ảnh lên Cloudflare R2`;
            }
        }
    }
}

// Handle multiple file upload
async function handleMultipleFileUpload() {
    if (selectedFiles.length < 2 || selectedFiles.length % 2 !== 0) {
        alert('Vui lòng chọn số lượng ảnh chẵn (ít nhất 2 ảnh)!');
        return;
    }
    
    // Tạo các cặp ảnh
    const pairs = [];
    for (let i = 0; i < selectedFiles.length; i += 2) {
        pairs.push({
            image1: selectedFiles[i],
            image2: selectedFiles[i + 1]
        });
    }
    
    uploadProgress.style.display = 'block';
    progressFill.style.width = '0%';
    progressText.textContent = '0%';
    
    let successCount = 0;
    let failCount = 0;
    
    for (let i = 0; i < pairs.length; i++) {
        const pair = pairs[i];
        const formData = new FormData();
        formData.append('upload_type', 'file');
        formData.append('image1', pair.image1);
        formData.append('image2', pair.image2);
        
        try {
            const progress = ((i + 1) / pairs.length) * 100;
            progressFill.style.width = progress + '%';
            progressText.textContent = Math.round(progress) + '% - Đang upload cặp ' + (i + 1) + '/' + pairs.length;
            
            const response = await fetch('/AlbumUD/upload.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                successCount++;
            } else {
                failCount++;
                console.error('Upload error:', result.error || 'Unknown error', result.details || '');
            }
        } catch (error) {
            console.error('Upload error:', error);
            failCount++;
        }
    }
    
    // Reset form
    selectedFiles = [];
    fileInputMultiple.value = '';
    updatePreviewMultiple();
    updateUploadButton();
    
    // Hide progress
    setTimeout(() => {
        uploadProgress.style.display = 'none';
        progressFill.style.width = '0%';
        progressText.textContent = '0%';
        
        if (successCount > 0) {
            alert(`Upload thành công ${successCount} cặp ảnh!${failCount > 0 ? '\nCó ' + failCount + ' cặp lỗi. Vui lòng kiểm tra console để xem chi tiết.' : ''}`);
        } else {
            alert('Upload thất bại! Vui lòng kiểm tra console để xem chi tiết lỗi.');
        }
    }, 1000);
}

// Handle URL upload
async function handleUrlUpload() {
    const url1 = urlInput1.value.trim();
    const url2 = urlInput2.value.trim();
    
    if (!url1 || !url2) {
        alert('Vui lòng nhập đầy đủ 2 link ảnh!');
        return;
    }
    
    // Validate URLs
    try {
        new URL(url1);
        new URL(url2);
    } catch (e) {
        alert('URL không hợp lệ!');
        return;
    }
    
    uploadProgress.style.display = 'block';
    progressFill.style.width = '0%';
    progressText.textContent = '0%';
    
    const formData = new FormData();
    formData.append('upload_type', 'url');
    formData.append('url1', url1);
    formData.append('url2', url2);
    
    try {
        progressFill.style.width = '50%';
        progressText.textContent = '50% - Đang xử lý...';
        
        const response = await fetch('upload.php', {
            method: 'POST',
            body: formData
        });
        
        // Kiểm tra response status
        if (!response.ok) {
            const errorText = await response.text();
            let errorData;
            try {
                errorData = JSON.parse(errorText);
            } catch (e) {
                errorData = { error: errorText || 'Lỗi không xác định' };
            }
            throw new Error(errorData.error || `HTTP Error: ${response.status}`);
        }
        
        const result = await response.json();
        
        if (result.success) {
            progressFill.style.width = '100%';
            progressText.textContent = '100% - Hoàn thành!';
            
            // Reset form
            urlInput1.value = '';
            urlInput2.value = '';
            previewUrl1.innerHTML = '';
            previewUrl1.classList.remove('has-image');
            previewUrl2.innerHTML = '';
            previewUrl2.classList.remove('has-image');
            
            // Hiển thị thông báo thành công
            setTimeout(() => {
                alert('Thêm ảnh thành công!');
                uploadProgress.style.display = 'none';
                progressFill.style.width = '0%';
                progressText.textContent = '0%';
                // Reload gallery nếu có
                if (typeof loadImages === 'function') {
                    loadImages();
                } else if (window.location.pathname.includes('index.php')) {
                    window.location.reload();
                }
            }, 1000);
        } else {
            throw new Error(result.error || 'Upload thất bại');
        }
    } catch (error) {
        console.error('Upload error:', error);
        let errorMessage = 'Lỗi khi upload ảnh từ URL';
        if (error.message) {
            errorMessage = error.message;
        }
        alert(errorMessage);
        uploadProgress.style.display = 'none';
        progressFill.style.width = '0%';
        progressText.textContent = '0%';
    }
}

