// State
let images = [];
let videos = [];

// DOM Elements
const postForm = document.getElementById('postForm');
const imageUrlInput = document.getElementById('imageUrlInput');
const videoUrlInput = document.getElementById('videoUrlInput');
const imagesList = document.getElementById('imagesList');
const videosList = document.getElementById('videosList');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    setupEventListeners();
    renderImagesList();
    renderVideosList();
});

// Event Listeners
function setupEventListeners() {
    postForm.addEventListener('submit', handleSubmit);
    
    // Enter key to add URL
    imageUrlInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            addImageUrl();
        }
    });
    
    videoUrlInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            addVideoUrl();
        }
    });
}

// Add image URL
function addImageUrl() {
    const url = imageUrlInput.value.trim();
    
    if (!url) {
        alert('Vui lòng nhập link ảnh!');
        return;
    }
    
    // Validate URL
    try {
        new URL(url);
    } catch (e) {
        alert('URL không hợp lệ!');
        return;
    }
    
    // Check if already exists
    if (images.includes(url)) {
        alert('Link ảnh này đã được thêm!');
        return;
    }
    
    images.push(url);
    imageUrlInput.value = '';
    renderImagesList();
}

// Remove image URL
function removeImage(index) {
    images.splice(index, 1);
    renderImagesList();
}

// Render images list
function renderImagesList() {
    if (images.length === 0) {
        imagesList.innerHTML = '<p class="empty-list">Chưa có ảnh nào</p>';
        return;
    }
    
    imagesList.innerHTML = images.map((url, index) => {
        return `
        <div class="item-card">
            <img src="${url}" alt="Image ${index + 1}" onerror="this.src='https://via.placeholder.com/150?text=Error'">
            <div class="item-info">
                <p class="item-url">${url.substring(0, 50)}${url.length > 50 ? '...' : ''}</p>
                <button type="button" class="btn-remove" onclick="removeImage(${index})">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
        `;
    }).join('');
}

// Add video URL
function addVideoUrl() {
    const url = videoUrlInput.value.trim();
    
    if (!url) {
        alert('Vui lòng nhập link video!');
        return;
    }
    
    // Validate URL
    try {
        new URL(url);
    } catch (e) {
        alert('URL không hợp lệ!');
        return;
    }
    
    // Check if already exists
    if (videos.includes(url)) {
        alert('Link video này đã được thêm!');
        return;
    }
    
    videos.push(url);
    videoUrlInput.value = '';
    renderVideosList();
}

// Remove video URL
function removeVideo(index) {
    videos.splice(index, 1);
    renderVideosList();
}

// Render videos list
function renderVideosList() {
    if (videos.length === 0) {
        videosList.innerHTML = '<p class="empty-list">Chưa có video nào</p>';
        return;
    }
    
    videosList.innerHTML = videos.map((url, index) => {
        return `
        <div class="item-card">
            <div class="item-preview">
                <i class="fas fa-video"></i>
            </div>
            <div class="item-info">
                <p class="item-url">${url.substring(0, 50)}${url.length > 50 ? '...' : ''}</p>
                <button type="button" class="btn-remove" onclick="removeVideo(${index})">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
        `;
    }).join('');
}

// Handle form submission
async function handleSubmit(e) {
    e.preventDefault();
    
    const albumId = document.getElementById('albumId').value;
    
    if (!albumId) {
        alert('Không tìm thấy album ID!');
        return;
    }
    
    // Validate có ít nhất 1 ảnh hoặc 1 video
    if (images.length === 0 && videos.length === 0) {
        alert('Vui lòng thêm ít nhất 1 ảnh hoặc 1 video!');
        return;
    }
    
    const tagsInput = document.getElementById('tags').value.trim();
    const tags = tagsInput ? tagsInput.split(',').map(tag => tag.trim()).filter(tag => tag) : [];
    
    const formData = {
        album_id: albumId,
        title: document.getElementById('postTitle').value.trim(),
        description: document.getElementById('postDescription').value.trim(),
        images: images,
        videos: videos,
        cosplayer: document.getElementById('cosplayer').value.trim(),
        character: document.getElementById('character').value.trim(),
        source: document.getElementById('source').value.trim(),
        tags: tags
    };
    
    try {
        const response = await fetch('albums_api.php?action=create_post', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(formData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Tạo post thành công!');
            window.location.href = `album_detail.php?id=${albumId}`;
        } else {
            alert('Lỗi: ' + (result.error || 'Không thể tạo post'));
        }
    } catch (error) {
        console.error('Create post error:', error);
        alert('Lỗi khi tạo post');
    }
}

// Make functions global
window.addImageUrl = addImageUrl;
window.removeImage = removeImage;
window.addVideoUrl = addVideoUrl;
window.removeVideo = removeVideo;
