// State
let scrapedData = null;
let currentJsonData = null;

// DOM Elements
const scraperForm = document.getElementById('scraperForm');
const cosplayUrl = document.getElementById('cosplayUrl');
const albumSelect = document.getElementById('albumSelect');
const resultBox = document.getElementById('resultBox');
const loadingSpinner = document.getElementById('loadingSpinner');
const errorMessage = document.getElementById('errorMessage');
const infoDisplay = document.getElementById('infoDisplay');
const infoGrid = document.getElementById('infoGrid');
const imagePreview = document.getElementById('imagePreview');
const jsonOutput = document.getElementById('jsonOutput');
const copyJsonBtn = document.getElementById('copyJsonBtn');
const downloadJsonBtn = document.getElementById('downloadJsonBtn');
const importBtn = document.getElementById('importBtn');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadAlbums();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    scraperForm.addEventListener('submit', handleScrape);
    copyJsonBtn.addEventListener('click', copyJson);
    downloadJsonBtn.addEventListener('click', downloadJson);
    importBtn.addEventListener('click', importToAlbum);
}

// Load albums for dropdown
async function loadAlbums() {
    try {
        const response = await fetch('scraper_api.php?action=list_albums');
        const albums = await response.json();
        
        albums.forEach(album => {
            const option = document.createElement('option');
            option.value = album.id;
            option.textContent = album.title;
            albumSelect.appendChild(option);
        });
    } catch (error) {
        console.error('Load albums error:', error);
    }
}

// Handle scrape
async function handleScrape(e) {
    e.preventDefault();
    
    const url = cosplayUrl.value.trim();
    
    if (!url) {
        showError('Vui lòng nhập URL');
        return;
    }
    
    // Show loading
    resultBox.style.display = 'block';
    loadingSpinner.style.display = 'block';
    infoDisplay.style.display = 'none';
    errorMessage.style.display = 'none';
    
    try {
        const response = await fetch('scraper_api.php?action=scrape', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ url: url })
        });
        
        const result = await response.json();
        
        if (result.error) {
            showError(result.error);
            return;
        }
        
        scrapedData = result;
        displayResults(result);
        
    } catch (error) {
        console.error('Scrape error:', error);
        showError('Lỗi khi lấy thông tin: ' + error.message);
    } finally {
        loadingSpinner.style.display = 'none';
    }
}

// Show error
function showError(message) {
    errorMessage.textContent = message;
    errorMessage.style.display = 'block';
    infoDisplay.style.display = 'none';
}

// Display results
function displayResults(data) {
    // Create JSON output
    const jsonData = {
        title: data.title || `${data.cosplayer || ''} cosplay ${data.character || ''}${data.source ? ' - ' + data.source : ''}`.trim() || 'Untitled Post',
        cosplayer: data.cosplayer || '',
        character: data.character || '',
        source: data.source || '',
        photo_count: data.photo_count || 0,
        video_count: data.video_count || 0,
        images: data.images || [],
        videos: data.videos || [],
        tags: data.tags || [],
        categories: data.categories || [],
        download_links: data.download_links || [],
        source_url: data.source_url || ''
    };
    
    currentJsonData = jsonData;
    
    // Display info grid
    infoGrid.innerHTML = `
        <div class="info-item">
            <label>Cosplayer</label>
            <div class="value">${escapeHtml(jsonData.cosplayer || 'N/A')}</div>
        </div>
        <div class="info-item">
            <label>Character</label>
            <div class="value">${escapeHtml(jsonData.character || 'N/A')}</div>
        </div>
        <div class="info-item">
            <label>Source</label>
            <div class="value">${escapeHtml(jsonData.source || 'N/A')}</div>
        </div>
        <div class="info-item">
            <label>Số ảnh</label>
            <div class="value">${jsonData.photo_count}</div>
        </div>
        <div class="info-item">
            <label>Số video</label>
            <div class="value">${jsonData.video_count}</div>
        </div>
        <div class="info-item">
            <label>Tags</label>
            <div class="value">${jsonData.tags.length > 0 ? jsonData.tags.join(', ') : 'N/A'}</div>
        </div>
        <div class="info-item">
            <label>Categories</label>
            <div class="value">${jsonData.categories.length > 0 ? jsonData.categories.join(', ') : 'N/A'}</div>
        </div>
        <div class="info-item">
            <label>Download Links</label>
            <div class="value">${jsonData.download_links.length} links</div>
        </div>
    `;
    
    // Display images preview
    if (jsonData.images && jsonData.images.length > 0) {
        imagePreview.innerHTML = jsonData.images.slice(0, 12).map(img => `
            <div class="image-preview-item">
                <img src="${img}" alt="Preview" onerror="this.src='https://via.placeholder.com/150?text=Error'">
            </div>
        `).join('');
        
        if (jsonData.images.length > 12) {
            imagePreview.innerHTML += `<div class="info-item" style="grid-column: 1 / -1; text-align: center; padding: 20px;">
                <p>Và ${jsonData.images.length - 12} ảnh khác...</p>
            </div>`;
        }
    } else {
        imagePreview.innerHTML = '<p class="empty-list">Không tìm thấy ảnh</p>';
    }
    
    // Display JSON
    jsonOutput.textContent = JSON.stringify(jsonData, null, 2);
    
    // Show display
    infoDisplay.style.display = 'block';
}

// Copy JSON
function copyJson() {
    if (!currentJsonData) return;
    
    const jsonString = JSON.stringify(currentJsonData, null, 2);
    navigator.clipboard.writeText(jsonString).then(() => {
        copyJsonBtn.innerHTML = '<i class="fas fa-check"></i> Đã Copy!';
        setTimeout(() => {
            copyJsonBtn.innerHTML = '<i class="fas fa-copy"></i> Copy JSON';
        }, 2000);
    }).catch(err => {
        console.error('Copy failed:', err);
        alert('Lỗi khi copy. Vui lòng copy thủ công từ ô JSON.');
    });
}

// Download JSON
function downloadJson() {
    if (!currentJsonData) return;
    
    const jsonString = JSON.stringify(currentJsonData, null, 2);
    const blob = new Blob([jsonString], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `cosplay_${Date.now()}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

// Import to album
async function importToAlbum() {
    const albumId = albumSelect.value;
    
    if (!albumId) {
        alert('Vui lòng chọn album để import!');
        return;
    }
    
    if (!currentJsonData) {
        alert('Không có dữ liệu để import!');
        return;
    }
    
    // Prepare post data
    const postData = {
        album_id: albumId,
        title: currentJsonData.title,
        cosplayer: currentJsonData.cosplayer,
        character: currentJsonData.character,
        source: currentJsonData.source,
        images: currentJsonData.images || [],
        videos: currentJsonData.videos || [],
        tags: currentJsonData.tags || [],
        download_links: currentJsonData.download_links || [],
        photo_count: currentJsonData.photo_count || 0,
        video_count: currentJsonData.video_count || 0,
        source_url: currentJsonData.source_url || '',
        categories: currentJsonData.categories || []
    };
    
    if (postData.images.length === 0 && postData.videos.length === 0) {
        alert('Post phải có ít nhất 1 ảnh hoặc 1 video!');
        return;
    }
    
    try {
        importBtn.disabled = true;
        importBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Đang import...';
        
        const response = await fetch('albums_api.php?action=create_post', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(postData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Import thành công!');
            window.location.href = `album_detail.php?id=${albumId}`;
        } else {
            alert('Lỗi: ' + (result.error || 'Không thể import post'));
        }
    } catch (error) {
        console.error('Import error:', error);
        alert('Lỗi khi import: ' + error.message);
    } finally {
        importBtn.disabled = false;
        importBtn.innerHTML = '<i class="fas fa-upload"></i> Import vào Album';
    }
}

// Utility
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
