// State management
let images = [];
let currentImageIndex = 0;
let negativeStates = {}; // Lưu trạng thái negative của từng ảnh

// DOM Elements
const gallery = document.getElementById('gallery');
const loading = document.getElementById('loading');
const refreshBtn = document.getElementById('refreshBtn');
const imageModal = document.getElementById('imageModal');
const modalImage = document.getElementById('modalImage');
const modalClose = document.getElementById('modalClose');
const prevBtn = document.getElementById('prevBtn');
const nextBtn = document.getElementById('nextBtn');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadImages();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    // Refresh
    if (refreshBtn) {
        refreshBtn.addEventListener('click', loadImages);
    }
    
    // Modal
    if (modalClose) modalClose.addEventListener('click', closeModal);
    if (prevBtn) prevBtn.addEventListener('click', showPrevImage);
    if (nextBtn) nextBtn.addEventListener('click', showNextImage);
    
    if (imageModal) {
        imageModal.addEventListener('click', (e) => {
            if (e.target === imageModal) closeModal();
        });
    }
    
    // Keyboard navigation
    document.addEventListener('keydown', (e) => {
        if (imageModal && imageModal.classList.contains('show')) {
            if (e.key === 'Escape') closeModal();
            if (e.key === 'ArrowLeft') showPrevImage();
            if (e.key === 'ArrowRight') showNextImage();
        }
    });
}

// Load images from server
async function loadImages() {
    if (!gallery || !loading) return;
    
    loading.style.display = 'block';
    gallery.innerHTML = '';
    gallery.appendChild(loading);
    
    try {
        const response = await fetch('api.php?action=list');
        images = await response.json();
        renderGallery();
    } catch (error) {
        console.error('Load images error:', error);
        if (gallery) {
            gallery.innerHTML = '<div class="empty-state"><i class="fas fa-exclamation-triangle"></i><p>Lỗi khi tải ảnh</p></div>';
        }
    } finally {
        if (loading) loading.style.display = 'none';
    }
}

// Render gallery
function renderGallery() {
    if (!gallery) return;
    
    if (images.length === 0) {
        gallery.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-images"></i>
                <p>Chưa có ảnh nào. Hãy upload ảnh để bắt đầu!</p>
            </div>
        `;
        return;
    }
    
    gallery.innerHTML = images.map((image, index) => {
        // Xác định URL ảnh dựa trên trạng thái negative
        const imageUrl = negativeStates[image.id] ? image.negativeUrl : image.url;
        
        return `
        <div class="image-card ${negativeStates[image.id] ? 'negative' : ''}" data-id="${image.id}" data-index="${index}">
            <img src="${imageUrl}" alt="${image.filename}" loading="lazy" data-normal="${image.url}" data-negative="${image.negativeUrl}">
            <button class="eye-icon ${negativeStates[image.id] ? 'active' : ''}" 
                    onclick="toggleNegative('${image.id}')" 
                    title="${negativeStates[image.id] ? 'Tắt chế độ âm bản' : 'Bật chế độ âm bản'}">
                <i class="fas fa-eye"></i>
            </button>
            <button class="delete-btn" onclick="deleteImage('${image.id}')" title="Xóa ảnh">
                <i class="fas fa-trash"></i>
            </button>
        </div>
        `;
    }).join('');
    
    // Add click to open modal
    document.querySelectorAll('.image-card img').forEach((img, index) => {
        img.addEventListener('click', () => openModal(index));
    });
}

// Toggle negative/invert effect - Load ảnh negative thực sự
function toggleNegative(imageId) {
    const card = document.querySelector(`.image-card[data-id="${imageId}"]`);
    if (!card) return;
    
    const img = card.querySelector('img');
    const eyeIcon = card.querySelector('.eye-icon');
    const image = images.find(img => img.id === imageId);
    
    if (!image) return;
    
    if (negativeStates[imageId]) {
        // Tắt negative - chuyển về ảnh gốc
        negativeStates[imageId] = false;
        card.classList.remove('negative');
        if (eyeIcon) {
            eyeIcon.classList.remove('active');
            eyeIcon.title = 'Bật chế độ âm bản';
        }
        if (img) img.src = image.url; // Load ảnh gốc
    } else {
        // Bật negative - chuyển sang ảnh negative
        negativeStates[imageId] = true;
        card.classList.add('negative');
        if (eyeIcon) {
            eyeIcon.classList.add('active');
            eyeIcon.title = 'Tắt chế độ âm bản';
        }
        if (img) img.src = image.negativeUrl; // Load ảnh negative thực sự
    }
    
    // Update modal if open
    if (imageModal && imageModal.classList.contains('show')) {
        const currentImage = images[currentImageIndex];
        if (currentImage && currentImage.id === imageId) {
            if (negativeStates[imageId]) {
                modalImage.src = currentImage.negativeUrl;
            } else {
                modalImage.src = currentImage.url;
            }
        }
    }
}

// Delete image
async function deleteImage(imageId) {
    if (!confirm('Bạn có chắc muốn xóa ảnh này?')) {
        return;
    }
    
    try {
        const response = await fetch(`api.php?action=delete&id=${encodeURIComponent(imageId)}`, {
            method: 'POST'
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Remove from array
            images = images.filter(img => img.id !== imageId);
            delete negativeStates[imageId];
            renderGallery();
        } else {
            alert('Lỗi khi xóa ảnh: ' + (result.error || ''));
        }
    } catch (error) {
        console.error('Delete error:', error);
        alert('Lỗi khi xóa ảnh');
    }
}

// Modal functions
function openModal(index) {
    if (!imageModal || !modalImage) return;
    
    currentImageIndex = index;
    const image = images[index];
    
    if (!image) return;
    
    // Load ảnh dựa trên trạng thái negative
    if (negativeStates[image.id]) {
        modalImage.src = image.negativeUrl;
    } else {
        modalImage.src = image.url;
    }
    
    imageModal.classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    if (!imageModal) return;
    imageModal.classList.remove('show');
    document.body.style.overflow = '';
}

function showPrevImage() {
    if (images.length === 0 || !modalImage) return;
    currentImageIndex = (currentImageIndex - 1 + images.length) % images.length;
    const image = images[currentImageIndex];
    
    if (image) {
        if (negativeStates[image.id]) {
            modalImage.src = image.negativeUrl;
        } else {
            modalImage.src = image.url;
        }
    }
}

function showNextImage() {
    if (images.length === 0 || !modalImage) return;
    currentImageIndex = (currentImageIndex + 1) % images.length;
    const image = images[currentImageIndex];
    
    if (image) {
        if (negativeStates[image.id]) {
            modalImage.src = image.negativeUrl;
        } else {
            modalImage.src = image.url;
        }
    }
}

// Utility functions
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('vi-VN', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Make functions global for onclick handlers
window.toggleNegative = toggleNegative;
window.deleteImage = deleteImage;
