// State
let albums = [];
let filteredAlbums = [];

// DOM Elements
const albumsGrid = document.getElementById('albumsGrid');
const loading = document.getElementById('loading');
const searchInput = document.getElementById('searchInput');
const categoryFilter = document.getElementById('categoryFilter');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadAlbums();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    searchInput.addEventListener('input', filterAlbums);
    categoryFilter.addEventListener('change', filterAlbums);
}

// Load albums
async function loadAlbums() {
    loading.style.display = 'block';
    albumsGrid.innerHTML = '';
    albumsGrid.appendChild(loading);
    
    try {
        const response = await fetch('albums_api.php?action=list_albums');
        albums = await response.json();
        filteredAlbums = [...albums];
        renderAlbums();
    } catch (error) {
        console.error('Load albums error:', error);
        albumsGrid.innerHTML = '<div class="empty-state"><i class="fas fa-exclamation-triangle"></i><p>Lỗi khi tải albums</p></div>';
    } finally {
        loading.style.display = 'none';
    }
}

// Filter albums
function filterAlbums() {
    const searchTerm = searchInput.value.toLowerCase().trim();
    const category = categoryFilter.value;
    
    filteredAlbums = albums.filter(album => {
        const matchSearch = !searchTerm || 
            album.title.toLowerCase().includes(searchTerm) ||
            (album.description && album.description.toLowerCase().includes(searchTerm));
        
        const matchCategory = !category || album.category === category;
        
        return matchSearch && matchCategory;
    });
    
    renderAlbums();
}

// Render albums
function renderAlbums() {
    if (filteredAlbums.length === 0) {
        albumsGrid.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-folder-open"></i>
                <p>Không tìm thấy album nào. Hãy tạo album mới!</p>
            </div>
        `;
        return;
    }
    
    albumsGrid.innerHTML = filteredAlbums.map(album => {
        const thumbnail = album.thumbnail || 'https://via.placeholder.com/300x200?text=No+Image';
        const postCount = album.postCount || 0;
        const category = album.category ? `<span class="album-category">${album.category}</span>` : '';
        
        return `
        <div class="album-card" onclick="window.location.href='album_detail.php?id=${album.id}'">
            <div class="album-thumbnail">
                <img src="${thumbnail}" alt="${album.title}" onerror="this.src='https://via.placeholder.com/300x200?text=No+Image'">
                ${category}
            </div>
            <div class="album-info">
                <h3 class="album-title">${escapeHtml(album.title)}</h3>
                ${album.description ? `<p class="album-description">${escapeHtml(album.description.substring(0, 100))}${album.description.length > 100 ? '...' : ''}</p>` : ''}
                <div class="album-meta">
                    <span><i class="fas fa-images"></i> ${postCount} posts</span>
                    <span><i class="fas fa-calendar"></i> ${formatDate(album.createdDate)}</span>
                </div>
            </div>
        </div>
        `;
    }).join('');
}

// Utility functions
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('vi-VN', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}
