// State management
let images = [];
let currentImageIndex = 0;
let negativeStates = {}; // Lưu trạng thái negative của từng ảnh
let selectedFiles = { file1: null, file2: null };

// DOM Elements
const uploadBox1 = document.getElementById('uploadBox1');
const uploadBox2 = document.getElementById('uploadBox2');
const fileInput1 = document.getElementById('fileInput1');
const fileInput2 = document.getElementById('fileInput2');
const preview1 = document.getElementById('preview1');
const preview2 = document.getElementById('preview2');
const previewUrl1 = document.getElementById('previewUrl1');
const previewUrl2 = document.getElementById('previewUrl2');
const urlInput1 = document.getElementById('urlInput1');
const urlInput2 = document.getElementById('urlInput2');
const uploadBtn = document.getElementById('uploadBtn');
const uploadUrlBtn = document.getElementById('uploadUrlBtn');
const uploadProgress = document.getElementById('uploadProgress');
const progressFill = document.getElementById('progressFill');
const progressText = document.getElementById('progressText');
const gallery = document.getElementById('gallery');
const loading = document.getElementById('loading');
const refreshBtn = document.getElementById('refreshBtn');
const imageModal = document.getElementById('imageModal');
const modalImage = document.getElementById('modalImage');
const modalClose = document.getElementById('modalClose');
const prevBtn = document.getElementById('prevBtn');
const nextBtn = document.getElementById('nextBtn');
const tabBtns = document.querySelectorAll('.tab-btn');
const tabContents = document.querySelectorAll('.upload-tab-content');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadImages();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    // Tab switching
    tabBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            const tab = btn.dataset.tab;
            switchTab(tab);
        });
    });
    
    // File upload boxes
    uploadBox1.addEventListener('click', () => fileInput1.click());
    uploadBox2.addEventListener('click', () => fileInput2.click());
    
    uploadBox1.addEventListener('dragover', (e) => handleDragOver(e, uploadBox1));
    uploadBox1.addEventListener('drop', (e) => handleDrop(e, fileInput1, preview1, 'file1'));
    uploadBox1.addEventListener('dragleave', (e) => handleDragLeave(e, uploadBox1));
    
    uploadBox2.addEventListener('dragover', (e) => handleDragOver(e, uploadBox2));
    uploadBox2.addEventListener('drop', (e) => handleDrop(e, fileInput2, preview2, 'file2'));
    uploadBox2.addEventListener('dragleave', (e) => handleDragLeave(e, uploadBox2));
    
    fileInput1.addEventListener('change', (e) => handleFileSelect(e, preview1, 'file1'));
    fileInput2.addEventListener('change', (e) => handleFileSelect(e, preview2, 'file2'));
    
    // URL inputs
    urlInput1.addEventListener('input', () => previewImageFromUrl(urlInput1.value, previewUrl1));
    urlInput2.addEventListener('input', () => previewImageFromUrl(urlInput2.value, previewUrl2));
    
    // Upload buttons
    uploadBtn.addEventListener('click', handleFileUpload);
    uploadUrlBtn.addEventListener('click', handleUrlUpload);
    
    // Refresh
    refreshBtn.addEventListener('click', loadImages);
    
    // Modal
    modalClose.addEventListener('click', closeModal);
    prevBtn.addEventListener('click', showPrevImage);
    nextBtn.addEventListener('click', showNextImage);
    imageModal.addEventListener('click', (e) => {
        if (e.target === imageModal) closeModal();
    });
    
    // Keyboard navigation
    document.addEventListener('keydown', (e) => {
        if (imageModal.classList.contains('show')) {
            if (e.key === 'Escape') closeModal();
            if (e.key === 'ArrowLeft') showPrevImage();
            if (e.key === 'ArrowRight') showNextImage();
        }
    });
}

// Tab switching
function switchTab(tab) {
    tabBtns.forEach(btn => {
        if (btn.dataset.tab === tab) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
    
    tabContents.forEach(content => {
        if (content.id === tab + 'Tab') {
            content.classList.add('active');
        } else {
            content.classList.remove('active');
        }
    });
}

// Drag and Drop
function handleDragOver(e, box) {
    e.preventDefault();
    e.stopPropagation();
    box.style.background = '#e8ebff';
}

function handleDragLeave(e, box) {
    e.preventDefault();
    e.stopPropagation();
    box.style.background = '#f8f9ff';
}

function handleDrop(e, input, preview, fileKey) {
    e.preventDefault();
    e.stopPropagation();
    const box = e.currentTarget;
    box.style.background = '#f8f9ff';
    
    const files = e.dataTransfer.files;
    if (files.length > 0) {
        const file = files[0];
        if (file.type.startsWith('image/')) {
            // Tạo FileList giả để set vào input
            const dataTransfer = new DataTransfer();
            dataTransfer.items.add(file);
            input.files = dataTransfer.files;
            
            showPreview(file, preview);
            selectedFiles[fileKey] = file;
            updateUploadButton();
        }
    }
}

function handleFileSelect(e, preview, fileKey) {
    const file = e.target.files[0];
    if (file) {
        showPreview(file, preview);
        selectedFiles[fileKey] = file;
        updateUploadButton();
    }
}

function showPreview(file, previewBox) {
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = (e) => {
        previewBox.innerHTML = `<img src="${e.target.result}" alt="Preview">`;
        previewBox.classList.add('has-image');
    };
    reader.readAsDataURL(file);
}

function previewImageFromUrl(url, previewBox) {
    if (!url || !url.trim()) {
        previewBox.innerHTML = '';
        previewBox.classList.remove('has-image');
        return;
    }
    
    // Validate URL
    try {
        new URL(url);
        previewBox.innerHTML = `<img src="${url}" alt="Preview" onerror="this.parentElement.innerHTML=''; this.parentElement.classList.remove('has-image');">`;
        previewBox.classList.add('has-image');
    } catch (e) {
        previewBox.innerHTML = '';
        previewBox.classList.remove('has-image');
    }
}

function updateUploadButton() {
    if (selectedFiles.file1 && selectedFiles.file2) {
        uploadBtn.disabled = false;
    } else {
        uploadBtn.disabled = true;
    }
}

// Handle file upload
async function handleFileUpload() {
    if (!selectedFiles.file1 || !selectedFiles.file2) {
        alert('Vui lòng chọn đầy đủ 2 ảnh!');
        return;
    }
    
    uploadProgress.style.display = 'block';
    progressFill.style.width = '0%';
    progressText.textContent = '0%';
    
    const formData = new FormData();
    formData.append('upload_type', 'file');
    formData.append('image1', selectedFiles.file1);
    formData.append('image2', selectedFiles.file2);
    
    try {
        progressFill.style.width = '50%';
        progressText.textContent = '50%';
        
        const response = await fetch('/AlbumUD/upload.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            progressFill.style.width = '100%';
            progressText.textContent = '100%';
            
            // Reset form
            selectedFiles = { file1: null, file2: null };
            fileInput1.value = '';
            fileInput2.value = '';
            preview1.innerHTML = '';
            preview1.classList.remove('has-image');
            preview2.innerHTML = '';
            preview2.classList.remove('has-image');
            updateUploadButton();
            
            // Add to images array
            images.unshift(result.image);
            renderGallery();
            
            // Hide progress after delay
            setTimeout(() => {
                uploadProgress.style.display = 'none';
                progressFill.style.width = '0%';
                progressText.textContent = '0%';
            }, 1000);
        } else {
            alert('Lỗi: ' + (result.error || 'Upload thất bại'));
            uploadProgress.style.display = 'none';
        }
    } catch (error) {
        console.error('Upload error:', error);
        alert('Lỗi khi upload ảnh');
        uploadProgress.style.display = 'none';
    }
}

// Handle URL upload
async function handleUrlUpload() {
    const url1 = urlInput1.value.trim();
    const url2 = urlInput2.value.trim();
    
    if (!url1 || !url2) {
        alert('Vui lòng nhập đầy đủ 2 link ảnh!');
        return;
    }
    
    // Validate URLs
    try {
        new URL(url1);
        new URL(url2);
    } catch (e) {
        alert('URL không hợp lệ!');
        return;
    }
    
    uploadProgress.style.display = 'block';
    progressFill.style.width = '0%';
    progressText.textContent = '0%';
    
    const formData = new FormData();
    formData.append('upload_type', 'url');
    formData.append('url1', url1);
    formData.append('url2', url2);
    
    try {
        progressFill.style.width = '50%';
        progressText.textContent = '50%';
        
        const response = await fetch('/AlbumUD/upload.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            progressFill.style.width = '100%';
            progressText.textContent = '100%';
            
            // Reset form
            urlInput1.value = '';
            urlInput2.value = '';
            previewUrl1.innerHTML = '';
            previewUrl1.classList.remove('has-image');
            previewUrl2.innerHTML = '';
            previewUrl2.classList.remove('has-image');
            
            // Add to images array
            images.unshift(result.image);
            renderGallery();
            
            // Hide progress after delay
            setTimeout(() => {
                uploadProgress.style.display = 'none';
                progressFill.style.width = '0%';
                progressText.textContent = '0%';
            }, 1000);
        } else {
            alert('Lỗi: ' + (result.error || 'Upload thất bại'));
            uploadProgress.style.display = 'none';
        }
    } catch (error) {
        console.error('Upload error:', error);
        alert('Lỗi khi upload ảnh từ URL');
        uploadProgress.style.display = 'none';
    }
}

// Load images from server
async function loadImages() {
    loading.style.display = 'block';
    gallery.innerHTML = '';
    gallery.appendChild(loading);
    
    try {
        const response = await fetch('api.php?action=list');
        images = await response.json();
        renderGallery();
    } catch (error) {
        console.error('Load images error:', error);
        gallery.innerHTML = '<div class="empty-state"><i class="fas fa-exclamation-triangle"></i><p>Lỗi khi tải ảnh</p></div>';
    } finally {
        loading.style.display = 'none';
    }
}

// Render gallery
function renderGallery() {
    if (images.length === 0) {
        gallery.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-images"></i>
                <p>Chưa có ảnh nào. Hãy upload ảnh để bắt đầu!</p>
            </div>
        `;
        return;
    }
    
    gallery.innerHTML = images.map((image, index) => {
        // Xác định URL ảnh dựa trên trạng thái negative
        const imageUrl = negativeStates[image.id] ? image.negativeUrl : image.url;
        
        return `
        <div class="image-card ${negativeStates[image.id] ? 'negative' : ''}" data-id="${image.id}" data-index="${index}">
            <img src="${imageUrl}" alt="${image.filename}" loading="lazy" data-normal="${image.url}" data-negative="${image.negativeUrl}">
            <button class="eye-icon ${negativeStates[image.id] ? 'active' : ''}" 
                    onclick="toggleNegative('${image.id}')" 
                    title="${negativeStates[image.id] ? 'Tắt chế độ âm bản' : 'Bật chế độ âm bản'}">
                <i class="fas fa-eye"></i>
            </button>
            <button class="delete-btn" onclick="deleteImage('${image.id}')" title="Xóa ảnh">
                <i class="fas fa-trash"></i>
            </button>
            <div class="image-info">
                <p><i class="fas fa-file"></i> ${image.filename}</p>
                <p><i class="fas fa-calendar"></i> ${formatDate(image.uploadDate)}</p>
            </div>
        </div>
        `;
    }).join('');
    
    // Add click to open modal
    document.querySelectorAll('.image-card img').forEach((img, index) => {
        img.addEventListener('click', () => openModal(index));
    });
}

// Toggle negative/invert effect - Load ảnh negative thực sự
function toggleNegative(imageId) {
    const card = document.querySelector(`.image-card[data-id="${imageId}"]`);
    const img = card.querySelector('img');
    const eyeIcon = card.querySelector('.eye-icon');
    const image = images.find(img => img.id === imageId);
    
    if (!image) return;
    
    if (negativeStates[imageId]) {
        // Tắt negative - chuyển về ảnh gốc
        negativeStates[imageId] = false;
        card.classList.remove('negative');
        eyeIcon.classList.remove('active');
        eyeIcon.title = 'Bật chế độ âm bản';
        img.src = image.url; // Load ảnh gốc
    } else {
        // Bật negative - chuyển sang ảnh negative
        negativeStates[imageId] = true;
        card.classList.add('negative');
        eyeIcon.classList.add('active');
        eyeIcon.title = 'Tắt chế độ âm bản';
        img.src = image.negativeUrl; // Load ảnh negative thực sự
    }
    
    // Update modal if open
    if (imageModal.classList.contains('show')) {
        const currentImage = images[currentImageIndex];
        if (currentImage.id === imageId) {
            if (negativeStates[imageId]) {
                modalImage.src = currentImage.negativeUrl;
            } else {
                modalImage.src = currentImage.url;
            }
        }
    }
}

// Delete image
async function deleteImage(imageId) {
    if (!confirm('Bạn có chắc muốn xóa ảnh này?')) {
        return;
    }
    
    try {
        const response = await fetch(`api.php?action=delete&id=${encodeURIComponent(imageId)}`, {
            method: 'POST'
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Remove from array
            images = images.filter(img => img.id !== imageId);
            delete negativeStates[imageId];
            renderGallery();
        } else {
            alert('Lỗi khi xóa ảnh: ' + (result.error || ''));
        }
    } catch (error) {
        console.error('Delete error:', error);
        alert('Lỗi khi xóa ảnh');
    }
}

// Modal functions
function openModal(index) {
    currentImageIndex = index;
    const image = images[index];
    
    // Load ảnh dựa trên trạng thái negative
    if (negativeStates[image.id]) {
        modalImage.src = image.negativeUrl;
    } else {
        modalImage.src = image.url;
    }
    
    imageModal.classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    imageModal.classList.remove('show');
    document.body.style.overflow = '';
}

function showPrevImage() {
    if (images.length === 0) return;
    currentImageIndex = (currentImageIndex - 1 + images.length) % images.length;
    const image = images[currentImageIndex];
    
    if (negativeStates[image.id]) {
        modalImage.src = image.negativeUrl;
    } else {
        modalImage.src = image.url;
    }
}

function showNextImage() {
    if (images.length === 0) return;
    currentImageIndex = (currentImageIndex + 1) % images.length;
    const image = images[currentImageIndex];
    
    if (negativeStates[image.id]) {
        modalImage.src = image.negativeUrl;
    } else {
        modalImage.src = image.url;
    }
}

// Utility functions
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('vi-VN', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Make functions global for onclick handlers
window.toggleNegative = toggleNegative;
window.deleteImage = deleteImage;
